/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.core.network;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.californium.core.Utils;
import org.eclipse.californium.core.coap.CoAP;
import org.eclipse.californium.core.coap.EmptyMessage;
import org.eclipse.californium.core.coap.Message;
import org.eclipse.californium.core.coap.MessageFormatException;
import org.eclipse.californium.core.coap.Request;
import org.eclipse.californium.core.coap.Response;
import org.eclipse.californium.core.network.Endpoint;
import org.eclipse.californium.core.network.EndpointManager;
import org.eclipse.californium.core.network.EndpointObserver;
import org.eclipse.californium.core.network.Exchange;
import org.eclipse.californium.core.network.InMemoryMessageExchangeStore;
import org.eclipse.californium.core.network.Matcher;
import org.eclipse.californium.core.network.MessageExchangeStore;
import org.eclipse.californium.core.network.Outbox;
import org.eclipse.californium.core.network.TcpMatcher;
import org.eclipse.californium.core.network.UdpMatcher;
import org.eclipse.californium.core.network.config.NetworkConfig;
import org.eclipse.californium.core.network.interceptors.MessageInterceptor;
import org.eclipse.californium.core.network.serialization.DataParser;
import org.eclipse.californium.core.network.serialization.DataSerializer;
import org.eclipse.californium.core.network.serialization.TcpDataParser;
import org.eclipse.californium.core.network.serialization.TcpDataSerializer;
import org.eclipse.californium.core.network.serialization.UdpDataParser;
import org.eclipse.californium.core.network.serialization.UdpDataSerializer;
import org.eclipse.californium.core.network.stack.CoapStack;
import org.eclipse.californium.core.network.stack.CoapTcpStack;
import org.eclipse.californium.core.network.stack.CoapUdpStack;
import org.eclipse.californium.core.server.MessageDeliverer;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.CorrelationContext;
import org.eclipse.californium.elements.MessageCallback;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.eclipse.californium.elements.UDPConnector;
import org.eclipse.californium.elements.tcp.TcpConnector;

public class CoapEndpoint
implements Endpoint {
    private static final Logger LOGGER = Logger.getLogger(CoapEndpoint.class.getCanonicalName());
    private final CoapStack coapstack;
    private final Connector connector;
    private final NetworkConfig config;
    private final Matcher matcher;
    private final DataSerializer serializer;
    private final DataParser parser;
    private ScheduledExecutorService executor;
    private boolean started;
    private List<EndpointObserver> observers = new CopyOnWriteArrayList<EndpointObserver>();
    private List<MessageInterceptor> interceptors = new CopyOnWriteArrayList<MessageInterceptor>();
    private final MessageExchangeStore exchangeStore;

    public CoapEndpoint() {
        this(0);
    }

    public CoapEndpoint(int port) {
        this(new InetSocketAddress(port));
    }

    public CoapEndpoint(InetSocketAddress address) {
        this(address, NetworkConfig.getStandard());
    }

    public CoapEndpoint(NetworkConfig config) {
        this(new InetSocketAddress(0), config);
    }

    public CoapEndpoint(int port, NetworkConfig config) {
        this(new InetSocketAddress(port), config);
    }

    public CoapEndpoint(InetSocketAddress address, NetworkConfig config) {
        this(address, config, null);
    }

    public CoapEndpoint(InetSocketAddress address, NetworkConfig config, MessageExchangeStore exchangeStore) {
        this(CoapEndpoint.createUDPConnector(address, config), config, exchangeStore);
    }

    public CoapEndpoint(Connector connector, NetworkConfig config) {
        this(connector, config, null);
    }

    public CoapEndpoint(Connector connector, NetworkConfig config, MessageExchangeStore exchangeStore) {
        this.config = config;
        this.connector = connector;
        this.connector.setRawDataReceiver(new InboxImpl());
        this.exchangeStore = exchangeStore;
        if (connector instanceof TcpConnector) {
            this.matcher = new TcpMatcher(config);
            this.coapstack = new CoapTcpStack(config, new OutboxImpl());
            this.serializer = new TcpDataSerializer();
            this.parser = new TcpDataParser();
        } else {
            this.matcher = new UdpMatcher(config);
            this.coapstack = new CoapUdpStack(config, new OutboxImpl());
            this.serializer = new UdpDataSerializer();
            this.parser = new UdpDataParser();
        }
    }

    private static Connector createUDPConnector(InetSocketAddress address, NetworkConfig config) {
        UDPConnector c = new UDPConnector(address);
        c.setReceiverThreadCount(config.getInt("NETWORK_STAGE_RECEIVER_THREAD_COUNT"));
        c.setSenderThreadCount(config.getInt("NETWORK_STAGE_SENDER_THREAD_COUNT"));
        c.setReceiveBufferSize(config.getInt("UDP_CONNECTOR_RECEIVE_BUFFER"));
        c.setSendBufferSize(config.getInt("UDP_CONNECTOR_SEND_BUFFER"));
        c.setReceiverPacketSize(config.getInt("UDP_CONNECTOR_DATAGRAM_SIZE"));
        return c;
    }

    @Override
    public synchronized void start() throws IOException {
        if (this.started) {
            LOGGER.log(Level.FINE, "Endpoint at {0} is already started", this.getAddress());
            return;
        }
        if (!this.coapstack.hasDeliverer()) {
            this.setMessageDeliverer(new EndpointManager.ClientMessageDeliverer());
        }
        if (this.executor == null) {
            LOGGER.log(Level.CONFIG, "Endpoint [{0}] requires an executor to start, using default single-threaded daemon executor", this.getAddress());
            this.setExecutor(Executors.newSingleThreadScheduledExecutor(new Utils.DaemonThreadFactory("CoapEndpoint-" + this.connector.getAddress() + '#')));
            this.addObserver(new EndpointObserver(){

                @Override
                public void started(Endpoint endpoint) {
                }

                @Override
                public void stopped(Endpoint endpoint) {
                }

                @Override
                public void destroyed(Endpoint endpoint) {
                    CoapEndpoint.this.executor.shutdown();
                }
            });
        }
        if (this.exchangeStore == null) {
            this.matcher.setMessageExchangeStore(new InMemoryMessageExchangeStore(this.config));
        } else {
            this.matcher.setMessageExchangeStore(this.exchangeStore);
        }
        try {
            LOGGER.log(Level.INFO, "Starting endpoint at {0}", this.getAddress());
            this.started = true;
            this.matcher.start();
            this.connector.start();
            for (EndpointObserver obs : this.observers) {
                obs.started(this);
            }
            this.startExecutor();
        }
        catch (IOException e) {
            this.stop();
            throw e;
        }
    }

    private void startExecutor() {
        this.runInProtocolStage(new Runnable(){

            @Override
            public void run() {
            }
        });
    }

    @Override
    public synchronized void stop() {
        if (!this.started) {
            LOGGER.log(Level.INFO, "Endpoint at {0} is already stopped", this.getAddress());
        } else {
            LOGGER.log(Level.INFO, "Stopping endpoint at address {0}", this.getAddress());
            this.started = false;
            this.connector.stop();
            this.matcher.stop();
            for (EndpointObserver obs : this.observers) {
                obs.stopped(this);
            }
            this.matcher.clear();
        }
    }

    @Override
    public synchronized void destroy() {
        LOGGER.log(Level.INFO, "Destroying endpoint at address {0}", this.getAddress());
        if (this.started) {
            this.stop();
        }
        this.connector.destroy();
        this.coapstack.destroy();
        for (EndpointObserver obs : this.observers) {
            obs.destroyed(this);
        }
    }

    @Override
    public void clear() {
        this.matcher.clear();
    }

    @Override
    public synchronized boolean isStarted() {
        return this.started;
    }

    @Override
    public synchronized void setExecutor(ScheduledExecutorService executor) {
        this.executor = executor;
        this.coapstack.setExecutor(executor);
    }

    @Override
    public void addObserver(EndpointObserver observer) {
        this.observers.add(observer);
    }

    @Override
    public void removeObserver(EndpointObserver observer) {
        this.observers.remove(observer);
    }

    @Override
    public void addInterceptor(MessageInterceptor interceptor) {
        this.interceptors.add(interceptor);
    }

    @Override
    public void removeInterceptor(MessageInterceptor interceptor) {
        this.interceptors.remove(interceptor);
    }

    @Override
    public List<MessageInterceptor> getInterceptors() {
        return Collections.unmodifiableList(this.interceptors);
    }

    @Override
    public void sendRequest(final Request request) {
        this.runInProtocolStage(new Runnable(){

            @Override
            public void run() {
                CoapEndpoint.this.coapstack.sendRequest(request);
            }
        });
    }

    @Override
    public void sendResponse(final Exchange exchange, final Response response) {
        if (exchange.hasCustomExecutor()) {
            this.runInProtocolStage(new Runnable(){

                @Override
                public void run() {
                    CoapEndpoint.this.coapstack.sendResponse(exchange, response);
                }
            });
        } else {
            this.coapstack.sendResponse(exchange, response);
        }
    }

    @Override
    public void sendEmptyMessage(Exchange exchange, EmptyMessage message) {
        this.coapstack.sendEmptyMessage(exchange, message);
    }

    @Override
    public void setMessageDeliverer(MessageDeliverer deliverer) {
        this.coapstack.setDeliverer(deliverer);
    }

    @Override
    public InetSocketAddress getAddress() {
        return this.connector.getAddress();
    }

    @Override
    public NetworkConfig getConfig() {
        return this.config;
    }

    private void runInProtocolStage(final Runnable task) {
        this.executor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    task.run();
                }
                catch (Throwable t) {
                    LOGGER.log(Level.SEVERE, String.format("Exception in protocol stage thread: %s", t.getMessage()), t);
                }
            }
        });
    }

    private class InboxImpl
    implements RawDataChannel {
        private InboxImpl() {
        }

        @Override
        public void receiveData(final RawData raw) {
            if (raw.getAddress() == null) {
                throw new IllegalArgumentException("received message that does not have a source address");
            }
            if (raw.getPort() == 0) {
                throw new IllegalArgumentException("received message that does not have a source port");
            }
            CoapEndpoint.this.runInProtocolStage(new Runnable(){

                @Override
                public void run() {
                    InboxImpl.this.receiveMessage(raw);
                }
            });
        }

        private void receiveMessage(RawData raw) {
            Message msg = null;
            try {
                msg = CoapEndpoint.this.parser.parseMessage(raw);
                msg.setSource(raw.getAddress());
                msg.setSourcePort(raw.getPort());
                if (CoAP.isRequest(msg.getRawCode())) {
                    this.receiveRequest((Request)msg, raw);
                } else if (CoAP.isResponse(msg.getRawCode())) {
                    this.receiveResponse((Response)msg, raw);
                } else if (CoAP.isEmptyMessage(msg.getRawCode())) {
                    this.receiveEmptyMessage((EmptyMessage)msg, raw);
                } else {
                    LOGGER.log(Level.FINER, "Silently ignoring non-CoAP message from {0}", raw.getInetSocketAddress());
                }
            }
            catch (MessageFormatException e) {
                if (e.isConfirmable() && e.hasMid()) {
                    this.reject(raw, e);
                }
                LOGGER.log(Level.FINER, "discarding malformed message from [{0}]", raw.getInetSocketAddress());
            }
        }

        private void reject(RawData raw, MessageFormatException cause) {
            EmptyMessage rst = new EmptyMessage(CoAP.Type.RST);
            rst.setMID(cause.getMid());
            rst.setToken(new byte[0]);
            rst.setDestination(raw.getAddress());
            rst.setDestinationPort(raw.getPort());
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.sendEmptyMessage(rst);
            }
            CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serializeEmptyMessage(rst));
            LOGGER.log(Level.FINE, "rejected malformed message from [{0}], reason: {1}", new Object[]{raw.getInetSocketAddress(), cause.getMessage()});
        }

        private void reject(Message message) {
            EmptyMessage rst = EmptyMessage.newRST(message);
            rst.setToken(new byte[0]);
            for (MessageInterceptor messageInterceptor : CoapEndpoint.this.interceptors) {
                messageInterceptor.sendEmptyMessage(rst);
            }
            if (!rst.isCanceled()) {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serializeEmptyMessage(rst));
            }
        }

        private void receiveRequest(Request request, RawData raw) {
            Exchange exchange;
            request.setScheme(raw.isSecure() ? "coaps" : "coap");
            request.setSenderIdentity(raw.getSenderIdentity());
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.receiveRequest(request);
            }
            if (!request.isCanceled() && (exchange = CoapEndpoint.this.matcher.receiveRequest(request)) != null) {
                exchange.setEndpoint(CoapEndpoint.this);
                CoapEndpoint.this.coapstack.receiveRequest(exchange, request);
            }
        }

        private void receiveResponse(Response response, RawData raw) {
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.receiveResponse(response);
            }
            if (!response.isCanceled()) {
                Exchange exchange = CoapEndpoint.this.matcher.receiveResponse(response, raw.getCorrelationContext());
                if (exchange != null) {
                    exchange.setEndpoint(CoapEndpoint.this);
                    response.setRTT(System.currentTimeMillis() - exchange.getTimestamp());
                    CoapEndpoint.this.coapstack.receiveResponse(exchange, response);
                } else if (response.getType() != CoAP.Type.ACK) {
                    LOGGER.log(Level.FINE, "Rejecting unmatchable response from {0}", raw.getInetSocketAddress());
                    this.reject(response);
                }
            }
        }

        private void receiveEmptyMessage(EmptyMessage message, RawData raw) {
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.receiveEmptyMessage(message);
            }
            if (!message.isCanceled()) {
                if (message.getType() == CoAP.Type.CON || message.getType() == CoAP.Type.NON) {
                    LOGGER.log(Level.FINER, "responding to ping from {0}", raw.getInetSocketAddress());
                    this.reject(message);
                } else {
                    Exchange exchange = CoapEndpoint.this.matcher.receiveEmptyMessage(message);
                    if (exchange != null) {
                        exchange.setEndpoint(CoapEndpoint.this);
                        CoapEndpoint.this.coapstack.receiveEmptyMessage(exchange, message);
                    }
                }
            }
        }
    }

    private class OutboxImpl
    implements Outbox {
        private OutboxImpl() {
        }

        @Override
        public void sendRequest(final Exchange exchange, Request request) {
            this.assertMessageHasDestinationAddress(request);
            CoapEndpoint.this.matcher.sendRequest(exchange, request);
            for (MessageInterceptor messageInterceptor : CoapEndpoint.this.interceptors) {
                messageInterceptor.sendRequest(request);
            }
            if (!request.isCanceled()) {
                MessageCallback callback = new MessageCallback(){

                    @Override
                    public void onContextEstablished(CorrelationContext context) {
                        exchange.setCorrelationContext(context);
                    }
                };
                RawData message = CoapEndpoint.this.serializer.serializeRequest(request, callback);
                CoapEndpoint.this.connector.send(message);
            }
        }

        @Override
        public void sendResponse(Exchange exchange, Response response) {
            this.assertMessageHasDestinationAddress(response);
            CoapEndpoint.this.matcher.sendResponse(exchange, response);
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.sendResponse(response);
            }
            if (!response.isCanceled()) {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serializeResponse(response));
            }
        }

        @Override
        public void sendEmptyMessage(Exchange exchange, EmptyMessage message) {
            this.assertMessageHasDestinationAddress(message);
            CoapEndpoint.this.matcher.sendEmptyMessage(exchange, message);
            for (MessageInterceptor interceptor : CoapEndpoint.this.interceptors) {
                interceptor.sendEmptyMessage(message);
            }
            if (!message.isCanceled()) {
                CoapEndpoint.this.connector.send(CoapEndpoint.this.serializer.serializeEmptyMessage(message));
            }
        }

        private void assertMessageHasDestinationAddress(Message message) {
            if (message.getDestination() == null) {
                throw new IllegalArgumentException("Message has no destination address");
            }
            if (message.getDestinationPort() == 0) {
                throw new IllegalArgumentException("Message has no destination port");
            }
        }
    }
}

